#####################################################################
# The following packages need to be installed before running the app
# Having all these packages installed, select all and run
#####################################################################

# install.packages("shiny")
# install.packages("MASS")
# install.packages("corrplot")
# install.packages("lavaan")
# install.packages("dplyr")
# install.packages("knitr")
# install.packages("semTools")


######################################################
# Define UI for Reliability in SEM application
######################################################

library(shiny)

ui <- pageWithSidebar(
  
  #  Application title
  headerPanel("Reliability in SEM"),
  
  sidebarPanel(
    
    tags$h3("Data Generation"),
    
    # Horizontal line ----
    tags$hr(),
    
    sliderInput("nitem", 
                "Number of Items:", 
                min = 3, 
                max = 30, 
                value = 6,
                step = 1),
    
    br(),
    
    sliderInput("nob", 
                "Number of Observations:", 
                min = 30, 
                max = 500, 
                value = 300,
                step = 10),
    
    br(),
    
    numericInput("l_mu", 
                 "Mean of Standardized Factor Loadings:", 
                 0.67),
    # Include clarifying text ----
    helpText("Note: specify a mean loading between 0 and 1."),
    
    br(),
    
    numericInput("l_sd", 
                 "Standard Deviation of Standardized Factor Loadings:", 
                 0.15),
    
    br(),
    
    
    radioButtons("rating_scale",
                 label = "Rating Scale of Items:",
                 choices = list("Continuous", 
                                "5-Point",
                                "7-Point"),
                 selected = "Continuous",
                 inline = FALSE),
    
    br(),
    
    
    tags$h3("Fit the Model"),
    
    
    # Horizontal line ----
    tags$hr(),
    
    
    # #~~~ write a if else phrase to reflect this part for data generation
    radioButtons("structure",
                 label = "Measurement Structure:",
                 choices = list("Parallel Model",
                                "Tau-Equivalent Model",
                                "Congeneric Model"),
                 selected = "Tau-Equivalent Model",
                 inline = FALSE),
    
    
    tags$br(),
    h5("Created by:"),
    tags$a("ITEMS Module: Instructional Topics in Educational Measurement Series", 
           href="https://360.articulate.com/review/content/f748e8ba-f059-4998-8ebf-f04aa0e18156/review")),
  
  
  
  
  # Show the main display
  mainPanel(
    
    # Output: Header + HTML table with requested number of observations ----
    h4("Data View (First 6 Rows)"),
    tableOutput("dataView"),
    
    # Output: Header + HTML table with descriptives ----
    h4("Descriptive Statistics"),
    tableOutput("descriptives"),
    
    # Output: plot output ----
    h4("Correlations Between Items"),
    plotOutput("correlations"),
    
    # Output: Header + Verbatim text for reliability ----
    h4("Reliability"),
    verbatimTextOutput("reliability"),
    
    # Output: Verbatim text for parameter estimates----
    h4("Unstandardized Parameter Estimates"),
    verbatimTextOutput("param_est"),
    
    # Output: Verbatim text for model fit ----
    h4("Data-Model Fit"),
    verbatimTextOutput("fit_measures")
    
  )
)




###############################################
# Define Server Function
###############################################

# nitem, nob, mu, sd, 
# "rating_scale": cont_dat, five_point_dat, seven_point_dat
# "structure": para, tau, cong

server <- function(input, output) {
  
  # Function that generates scenarios and computes reliability indices.
  getReliability <- reactive({ 
    
    #-------------------------------------
    # Inputs
    #-------------------------------------
    
    # number of items: nitem
    nitem = input$nitem
    
    # number of observations: nob
    nob = input$nob
    
    # mean and standard deviation of factor loadings
    l_mu = input$l_mu
    l_sd = input$l_sd
    
    
    
    #-------------------------------------
    # Simulation
    #-------------------------------------
    
    #~~~~~~~~~~~~~~~~ generate continuous variables ~~~~~~~~~~~~~~~~~#
    
    library(MASS)
    set.seed(654321)
    
    # generates std. factor scores
    f =  rnorm(nob, 0, 1)
    
    # generate factor loadings from a truncated normal distribution (ranges from 0 to 1)
    l = qnorm(runif(nitem, pnorm(0, l_mu, l_sd), pnorm(1, l_mu, l_sd)), l_mu, l_sd)
    
    # generate factor scores
    f_dat = cbind(data.frame(f)[-1], replicate(nitem, f))
    
    # generate errors
    e_mu = rep(0, nitem)
    e_sigma = diag(1, nitem, nitem) 
    e_dat = data.frame(mvrnorm(nob, mu = e_mu, Sigma = e_sigma))
    
    
    
    #~~~ generate continuous data
    
    true = data.frame(t((as.vector(l) * t(f_dat))))
    error = data.frame(t(as.vector(sqrt(1 - l^2)) * t(e_dat)))
    cont_data = true + error
    detach("package:MASS", unload = TRUE) # detach "MASS" so "dplyr" can work correctly
    
    
    #~~~ categorize continuous items into x-point scale
    
    # X1 and X2 are respectively based on bell-shaped and uniform distributions
    
    F.CAT.BELL_UNI = function(ncat){
      
      # X1 is based on bell-shaped distribution
      zscores = seq(from = -3, to = 3, by = 6/ncat)
      X1 = cut(cont_data[, "X1"],
               breaks = c(quantile(cont_data[, "X1"], probs = c(0, pnorm(zscores)[-length(pnorm(zscores))][-1], 1))),
               labels = seq(from = 1, to = ncat, by = 1),
               include.lowest = TRUE)
      
      # X2 is based on uniform distribution
      X2 = cut(cont_data[, "X2"],
               breaks = c(quantile(cont_data[, "X2"], probs = seq(from = 0, to = 1, by = 1/ncat))),
               labels = seq(from = 1, to = ncat, by = 1),
               include.lowest = TRUE)
      data.frame(X1, X2)
    }
    
    
    # the rest of the categorical variables are based on random cutpoints
    
    F.CAT.RANDOM = function(x, ncat){
      xnew = cut(x,
                 breaks = c(quantile(x, probs = c(0, runif(n = ncat-1, min = 0.01, max = 0.99), 1))),
                 labels = seq(from = 1, to = ncat, by = 1),
                 include.lowest = TRUE)
    }
    
    
    # lapply wouldn't work for the 3-item condition, in which only 1 variable has random underlying scale
    
    if(nitem > 3){
      five_point_data = cbind(F.CAT.BELL_UNI(ncat = 5), data.frame(lapply(as.list(cont_data[, -c(1:2)]), F.CAT.RANDOM, ncat = 5)))
      seven_point_data = cbind(F.CAT.BELL_UNI(ncat = 7), data.frame(lapply(as.list(cont_data[, -c(1:2)]), F.CAT.RANDOM, ncat = 7)))
    } else{
      five_point_data = cbind(F.CAT.BELL_UNI(ncat = 5), data.frame(X3 = F.CAT.RANDOM(cont_data[, -c(1:2)], ncat = 5)))
      seven_point_data = cbind(F.CAT.BELL_UNI(ncat = 7), data.frame(X3 = F.CAT.RANDOM(cont_data[, -c(1:2)], ncat = 7)))
    }
    
    
    
    if(input$rating_scale == "Continuous"){
      mydata0 = cont_data
    } else if(input$rating_scale == "5-Point"){
      mydata0 = five_point_data
    } else{
      mydata0 = seven_point_data
    }
    
    
    
    #~~~ statistics of the data
    
    library(knitr) # use the kable function
    
    mydata = data.frame(cbind(f, sapply(mydata0, as.numeric)))
    headOfData = round(head(mydata[-1]), 3)
    descriptives0 = data.frame(sapply(mydata[-1], function(x) list(mean = mean(x),
                                                                   sd = sd(x))))
    descriptives1 = round(data.frame(sapply(descriptives0, function(x) as.numeric(as.character(x)))), 2)
    rownames(descriptives1) = c("Mean", "SD")
    descriptives = descriptives1
    
    correlation = cor(mydata[-1])
    library(corrplot)
    
    
    
    #~~~ run an SEM model with "lavaan"
    
    # specify the CFA model
    
    library(lavaan)
    library(dplyr) # detach MASS before running any dplyr functions
    
    if(input$structure == "Congeneric Model"){
      cfamodel = paste("f", "=~", paste(t(as.matrix(colnames(mydata0))), collapse = "+"), sep = "")
    } else if(input$structure == "Tau-Equivalent Model"){
      cfamodel = paste("f", "=~", paste("v1*", t(as.matrix(colnames(mydata0))), collapse = "+"), sep = "")
    } else{
      mod = paste("f", "=~", paste("v1*", t(as.matrix(colnames(mydata0))), collapse = "+"), sep = "")
      constraints = paste(t(as.matrix(colnames(mydata0))), " ~~ f1*", t(as.matrix(colnames(mydata0))), sep = "", collapse = ";")
      # put the two parts together
      cfamodel = paste(mod, ";", constraints) 
    }
    
    fit <- cfa(model = cfamodel, data = mydata, std.lv = TRUE) 
    fit_measures <- as.data.frame(t(as.matrix(fitMeasures(fit, c("df", "chisq", "aic", "bic", "srmr", "rmsea", "cfi"))))) %>%
      kable(digits = 3, format = "pandoc")
    param = parameterEstimates(fit, standardized = TRUE)
    
    
    l_est = param %>% 
      filter(op == "=~") %>% 
      select(Item = rhs, Loading = est, 'SE(loading)' = se) 
    var_est = param[-nrow(param)] %>% 
      filter(op == "~~" & lhs != "f") %>% 
      select(Item = rhs,'Error Variance' = est, 'SE(variance)' = se) 
    
    param_est = cbind(l_est, var_est[, -1]) %>%
      kable(digits = 3, format ="pandoc", caption = NULL)
    
    
    #~~~ calculate the results
    
    library(semTools)
    
    # print alpha and omega
    alpha = sprintf("%.3f", round(reliability(fit)[1,1], 3))
    omega = sprintf("%.3f", round(reliability(fit)[2,1], 3))
    rel0 = data.frame(alpha = alpha, omega = omega, row.names = "")
    rel = kable(rel0, digits = 3, format = "pandoc", row.names = FALSE)
    
    comboResults = list(headOfData = headOfData, descriptives = descriptives, correlation = correlation, 
                        rel = rel, param_est = param_est, fit_measures = fit_measures, mydata = mydata)
    comboResults
  })
  
  
  # return the results and present them in output
  
  output$dataView <- renderTable({
    combo = getReliability()
    combo$headOfData
  },
  include.rownames = FALSE)
  
  output$descriptives <- renderTable({
    combo = getReliability()
    combo$descriptives
  },
  include.rownames = TRUE) 
  
  output$correlations <- renderPlot({
    combo = getReliability()
    corrplot(cor(combo$mydata[-1]), method = "circle", diag = FALSE, type = "lower") # plot matrix
  }) 
  
  output$reliability <- renderPrint({
    combo = getReliability()
    combo$rel
  })
  
  output$param_est <- renderPrint({
    combo = getReliability()
    combo$param_est
  })
  
  output$fit_measures <- renderPrint({
    combo = getReliability()
    combo$fit_measures
  })
}




# Create Shiny app ----
shinyApp(ui = ui, server = server)

