library(shiny)

######################################################
# Define UI for Reliability in SEM application
######################################################

ui <- pageWithSidebar(
  
  #  Application title
  headerPanel("Reliability in SEM"),
  
  sidebarPanel(
    
    tags$h3("Data Generation"),
    
    # Horizontal line ----
    tags$hr(),
    
    sliderInput("nitem", 
                "Number of Items:", 
                min = 3, 
                max = 30, 
                value = 6,
                step = 1),
    
    sliderInput("nob", 
                "Number of Observations:", 
                min = 30, 
                max = 500, 
                value = 300,
                step = 1),
    
    numericInput("l_mu", 
                 "Mean of Factor Loadings:", 
                 0.67),
    
    numericInput("l_sd", 
                 "Standard Deviation of Factor Loadings:", 
                 0.15),
    
    
    radioButtons("rating_scale",
                 label = "Rating Scale of Items:",
                 choices = list("Continuous", 
                              "Likert",
                              "7-Point"),
                 selected = "Continuous",
                 inline = FALSE),
    
    # # Include clarifying text ----
    # helpText("Note: while the data view will show only the specified",
    #          "number of observations, the summary will still be based",
    #          "on the full dataset."),
    

    
    # br(),
    
    # selectInput("regression", "regression:",
    #             list("y~x", 
    #                  "y~x2",
    #                  "y~expx",
    #                  "y~x+x2",
    #                  "y~x+expx",
    #                  "y~x+x2+expx",
    #                  "y~x+x2+x3+x4+x5"
    #             )),
    
    
    # #~~~ write a if else phrase to reflect this part for data generation
    radioButtons("structure",
                 label = "Measurement Structure:",
                 choices = list("Parallel Model",
                                "Tau-Equivalent Model",
                                "Congeneric Model"),
                 selected = "Tau-Equivalent Model",
                 inline = FALSE),
    
    tags$br(),
    h5("Created by:"),
    tags$a("ITEMS Module: Instructional Topics in Educational Measurement Series", 
           href="https://360.articulate.com/review/content/f748e8ba-f059-4998-8ebf-f04aa0e18156/review")),
                
  
  
  
  # Show the main display
  mainPanel(
    
    # Output: Header + HTML table with requested number of observations ----
    h4("Data View (First 6 Rows)"),
    tableOutput("dataView"),
    
    # Output: Header + HTML table with descriptives ----
    h4("Descriptive Statistics"),
    tableOutput("descriptives"),
    
    # # Output: plot output ----
    # h4("Correlations between Items"),
    # plotOutput("correlations"),
    
    # Output: Header + Verbatim text for reliability ----
    h4("Reliability"),
    verbatimTextOutput("reliability"),
    
    # Output: Verbatim text for parameter estimates----
    h4("Parameter Estimates"),
    verbatimTextOutput("param_est"),
    
    # Output: Verbatim text for model fit ----
    h4("Data-Model Fit"),
    verbatimTextOutput("fit_measures")
    
  )
)




###############################################
# Define Server Function
###############################################

# nitem, nob, mu, sd, 
# "rating_scale": cont_dat, likert_dat,
# "structure": para, tau, cong

server <- function(input, output) {
  
  # Function that generates scenarios and computes reliability indices.
  getReliability <- reactive({ 
    
    #-------------------------------------
    # Inputs
    #-------------------------------------
    
    # number of items: nitem
    nitem = input$nitem
    
    # number of observations: nob
    nob = input$nob
    
    # mean and standard deviation of factor loadings
    l_mu = input$l_mu
    l_sd = input$l_sd
    
    
    
    # # measurement structure: para, tau, cong
    # structure = switch(input$structure,
    #                    "para" = para,
    #                    "tau" = tau,
    #                    "cong" = cong)
    
    
    #-------------------------------------
    # Simulation
    #-------------------------------------
    
    #~~~~~~~~~~~~~~~~ generate continuous variables ~~~~~~~~~~~~~~~~~#
    
    # # to test the simulation syntax
    # 
    # nitem = 6
    # nob = 500
    # l_mu = 0.67
    # l_sd = 0.15

    library(MASS)
    set.seed(654321)
    
    # the scale of factor scores does not matter
    f =  rnorm(nob, 6, 1)
    
    # generage factor loadings from a truncated normal distribution (ranges from 0 to 1)
    l = qnorm(runif(nitem, pnorm(0, l_mu, l_sd), pnorm(1, l_mu, l_sd)), l_mu, l_sd)
    
    # generate factor scores
    f_dat = cbind(data.frame(f)[-1], replicate(nitem, f))
    
    # generate errors
    e_mu = rep(0, nitem)
    e_sigma = diag(1, nitem, nitem)
    e_dat = data.frame(mvrnorm(nob, mu = e_mu, Sigma = e_sigma))
    
    
    
    #~~~ generate continuous data
    
    true = data.frame(t((as.vector(l) * t(f_dat))))
    error = data.frame(t(as.vector(sqrt(1 - l^2)) * t(e_dat)))
    cont_data = true + error
    detach("package:MASS", unload = TRUE) # detach "MASS" so "dplyr" can work correctly
    
    
    #~~~ categorize continuous items into x-point scale
    
    # X1 and X2 are respectively based on bell-shaped and uniform distributions
    
    F.CAT.BELL_UNI = function(ncat){
      
      # X1 is based on bell-shaped distribution
      zscores = seq(from = -3, to = 3, by = 6/ncat)
      X1 = cut(cont_data[, "X1"],
               breaks = c(quantile(cont_data[, "X1"], probs = c(0, pnorm(zscores)[-length(pnorm(zscores))][-1], 1))),
               labels = seq(from = 1, to = ncat, by = 1),
               include.lowest = TRUE)
      
      # X2 is based on uniform distribution
      X2 = cut(cont_data[, "X2"],
               breaks = c(quantile(cont_data[, "X2"], probs = seq(from = 0, to = 1, by = 1/ncat))),
               labels = seq(from = 1, to = ncat, by = 1),
               include.lowest = TRUE)
      data.frame(X1, X2)
    }
   

    # the rest of categorical variables are based on random cutpoints
    
    F.CAT.RANDOM = function(x, ncat){
      xnew = cut(x,
                 breaks = c(quantile(x, probs = c(0, runif(n = ncat-1, min = 0.01, max = 0.99), 1))),
                 labels = seq(from = 1, to = ncat, by = 1),
                 include.lowest = TRUE)
    }
    
    
    likert_data = cbind(F.CAT.BELL_UNI(ncat = 5), data.frame(lapply(as.list(cont_data[, -c(1:2)]), F.CAT.RANDOM, ncat = 5)))
    seven_point_data = cbind(F.CAT.BELL_UNI(ncat = 7), data.frame(lapply(as.list(cont_data[, -c(1:2)]), F.CAT.RANDOM, ncat = 7)))
    
    
    
    # F.CAT.BELL = function(x, ncat){
    #   xnew = cut(x,
    #              zscores = seq(from = -3, to = 3, by = 6/ncat),
    #              probs0 = pnorm(zscores),
    #              probs = c(0, probs, 1),
    #              breaks = c(quantile(x, probs = c(0, probs0[-length(probs0)][-1], 1))),
    #              labels = seq(from = 1, to = ncat, by = 1),
    #              include.lowest = TRUE)
    # }
    
    
    # both switch and if-else statement work
    
    # item scale: cont, likert
    # mydata0 = switch(input$rating_scale,
    #                       "Continuous" = cont_data,
    #                       "Likert" = likert_data)
    
    
    
    
    
    if(input$rating_scale == "Continuous"){
      mydata0 = cont_data
    } else if(input$rating_scale == "Likert"){
      mydata0 = likert_data
    } else{
      mydata0 = seven_point_data
    }
    
    
    
    #~~~ statistics of the data
    
    F.ROUNDdf <- function(x, digits) {
      # round all numeric variables
      # x: data frame 
      # digits: number of digits to round
      x <- sapply(x, as.numeric)
      numeric_columns <- sapply(x, mode) == 'numeric'
      x[numeric_columns] <-  round(x[numeric_columns], digits)
      x
    }
    
    # rating_scale = likert_dat # mydata will be selected based on which scale is selected (cont_dat or likert_dat)
    mydata = data.frame(cbind(f, sapply(mydata0, as.numeric)))
    headOfData = round(head(mydata), 3)
    descriptives0 = data.frame(sapply(mydata[-1], function(x) list(mean = mean(x),
                                                               sd = sd(x))))
    
    descriptives = F.ROUNDdf(descriptives0, 4)
    rownames(descriptives) = c("Mean", "SD")
    
    library('corrplot') #package corrplot
    correlation = cor(mydata[-1])
    correlation_plot = corrplot(correlation, method = "number", diag = FALSE) # plot matrix
    
    
    
    #~~~ run an SEM model with "lavaan"
    
    # specify the CFA model
    
    library(lavaan)
    library(dplyr) # detach MASS before running any dplyr functions
    library(knitr) # use the kable function
    
    if(input$structure == "Congeneric Model"){
      cfamodel = paste("f", "=~", paste(t(as.matrix(colnames(mydata0))), collapse = "+"), sep = "")
    } else if(input$structure == "Tau-Equivalent Model"){
      cfamodel = paste("f", "=~", paste("v1*", t(as.matrix(colnames(mydata0))), collapse = "+"), sep = "")
    } else{
      mod = paste("f", "=~", paste("v1*", t(as.matrix(colnames(mydata0))), collapse = "+"), sep = "")
      constraints = paste(t(as.matrix(colnames(mydata0))), " ~~ f1*", t(as.matrix(colnames(mydata0))), sep = "", collapse = ";")
      # put the two parts together
      cfamodel = paste(mod, ";", constraints) 
    }
    
    fit <- cfa(model = cfamodel, data = mydata, std.lv=TRUE) # std.lv is to constrain f to have a mean of 0 and sd of 1
    fit_measures <- round(fitMeasures(fit, c("df", "chisq", "aic", "bic", "srmr", "rmsea", "cfi")), 2)
    param = parameterEstimates(fit, standardized = TRUE)
    # summary(fit, fit.measures = TRUE, standardized = TRUE)
    
    l_est = param %>% 
      filter(op == "=~") %>% 
      select(Item = rhs, Loading = est, 'SE(loading)' = se) 
    var_est = param[-nrow(param)] %>% 
      filter(op == "~~" & lhs != "f") %>% 
      select(Item = rhs,'Error Variance' = est, 'SE(variance)' = se) 
    
    param_est = cbind(l_est, var_est[,-1]) %>%
      kable(digits = 3, format ="pandoc", caption = NULL)
    
    
    
    # this is a tau-equivalent model
    # cfa(model = 'f=~v1* X1+v1* X2+v1* X3+v1* X4+v1* X5+v1* X6', data = mydata)
    
    # this is a parallel model; ";" = line break
    # cfa(model = 'f=~v1* X1+v1* X2+v1* X3+v1* X4+v1* X5+v1* X6
    #        X1 ~~ f1*X1
    #        X2 ~~ f1*X2
    #        X3 ~~ f1*X3
    #        X4 ~~ f1*X4
    #        X5 ~~ f1*X5
    #        X6 ~~ f1*X6', 
    #        data = mydata)
    # cfa(model = 'f=~v1* X1+v1* X2+v1* X3+v1* X4+v1* X5+v1* X6;X1 ~~ f1*X1;X2 ~~ f1*X2;X3 ~~ f1*X3;X4 ~~ f1*X4;X5 ~~ f1*X5;X6 ~~ f1*X6', 
    #     data = mydata)

    
    #~~~ calculate the results
    
    library(semTools)
    # print alpha and omega
    alpha = round(reliability(fit)[1,1], 2)
    omega = round(reliability(fit)[2,1], 2)
    rel = data.frame(alpha = alpha, omega = omega, row.names = "")
    
    comboResults = list(headOfData = headOfData, descriptives = descriptives, correlation = correlation, 
                        rel = rel, param_est = param_est, fit_measures = fit_measures)
    comboResults
  })

    
  # return the results into output
  
  output$dataView <- renderTable({
    combo = getReliability()
    combo$headOfData
  },
  include.rownames = FALSE)
   
  output$descriptives <- renderTable({
    combo = getReliability()
    combo$descriptives
  },
  include.rownames = TRUE) 
  
  output$correlations <- renderPlot({
    combo = input$getReliability()
    corrplot(combo$correlation, method = "number", diag = FALSE) # plot matrix
  }) 
    
  output$reliability <- renderPrint({
    combo = getReliability()
    combo$rel
  })

  output$param_est <- renderPrint({
    combo = getReliability()
    combo$param_est
  })
  
  output$fit_measures <- renderPrint({
    combo = getReliability()
    combo$fit_measures
  })
}




# Create Shiny app ----
shinyApp(ui = ui, server = server)

